"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = __importDefault(require("assert"));
const path_1 = __importDefault(require("path"));
const typescript_1 = __importDefault(require("typescript"));
const documents_1 = require("../../../../src/lib/documents");
const ls_config_1 = require("../../../../src/ls-config");
const plugins_1 = require("../../../../src/plugins");
const WorkspaceSymbolProvider_1 = require("../../../../src/plugins/typescript/features/WorkspaceSymbolProvider");
const utils_1 = require("../../../../src/utils");
const test_utils_1 = require("../test-utils");
const testDir = path_1.default.join(__dirname, '..');
describe('WorkspaceSymbolsProvider', function () {
    (0, test_utils_1.serviceWarmup)(this, testDir, (0, utils_1.pathToUrl)(testDir));
    function getFullPath(filename) {
        return path_1.default.join(testDir, 'testfiles', 'workspace-symbols', filename);
    }
    function getUri(filename) {
        return (0, utils_1.pathToUrl)(getFullPath(filename));
    }
    function setup(filename) {
        const docManager = new documents_1.DocumentManager((textDocument) => new documents_1.Document(textDocument.uri, textDocument.text));
        const lsConfigManager = new ls_config_1.LSConfigManager();
        const lsAndTsDocResolver = new plugins_1.LSAndTSDocResolver(docManager, [(0, utils_1.pathToUrl)(testDir)], lsConfigManager);
        const provider = new WorkspaceSymbolProvider_1.WorkspaceSymbolsProviderImpl(lsAndTsDocResolver, lsConfigManager);
        const filePath = getFullPath(filename);
        const document = docManager.openClientDocument({
            uri: (0, utils_1.pathToUrl)(filePath),
            text: typescript_1.default.sys.readFile(filePath)
        });
        return { provider, document, docManager, lsAndTsDocResolver };
    }
    it('should return workspace symbols', async () => {
        const { provider, document, lsAndTsDocResolver } = setup('workspace-symbols.svelte');
        await lsAndTsDocResolver.getLSAndTSDoc(document);
        const symbols = await provider.getWorkspaceSymbols('longName');
        assert_1.default.deepStrictEqual(symbols, [
            {
                containerName: 'script',
                kind: 12,
                location: {
                    range: {
                        end: {
                            character: 5,
                            line: 3
                        },
                        start: {
                            character: 4,
                            line: 2
                        }
                    },
                    uri: getUri('workspace-symbols.svelte')
                },
                name: 'longLongName()',
                tags: undefined
            },
            {
                containerName: '',
                kind: 11,
                location: {
                    range: {
                        end: {
                            character: 1,
                            line: 5
                        },
                        start: {
                            character: 0,
                            line: 3
                        }
                    },
                    uri: getUri('imported.ts')
                },
                name: 'longLongName2',
                tags: [1]
            },
            {
                containerName: 'longLongName2',
                kind: 8,
                location: {
                    range: {
                        end: {
                            character: 26,
                            line: 4
                        },
                        start: {
                            character: 4,
                            line: 4
                        }
                    },
                    uri: getUri('imported.ts')
                },
                name: 'longLongName3',
                tags: undefined
            },
            {
                containerName: undefined,
                kind: 13,
                location: {
                    range: {
                        end: {
                            character: 28,
                            line: 8
                        },
                        start: {
                            character: 15,
                            line: 8
                        }
                    },
                    uri: getUri('workspace-symbols.svelte')
                },
                name: 'longLongName4',
                tags: undefined
            }
        ]);
    });
    it('filter out generated symbols', async () => {
        const { provider, document, lsAndTsDocResolver } = setup('workspace-symbols.svelte');
        await lsAndTsDocResolver.getLSAndTSDoc(document);
        const symbols = await provider.getWorkspaceSymbols('_');
        assert_1.default.deepStrictEqual(
        // Filter out the generated component class/const/type.
        // The unfiltered result is slightly different in svelte 4 and svelte 5,
        // and there is a maxResultCount limit, so it's not always present.
        onlyInWorkspaceSymbolsDir(symbols)?.filter((v) => v.name !== 'WorkspaceSymbols__SvelteComponent_'), []);
        const symbols2 = await provider.getWorkspaceSymbols('$');
        assert_1.default.deepStrictEqual(onlyInWorkspaceSymbolsDir(symbols2), []);
    });
    function onlyInWorkspaceSymbolsDir(symbols) {
        return symbols?.filter((f) => f.location.uri.includes('workspace-symbols'));
    }
});
//# sourceMappingURL=WorkspaceSymbolsProvider.test.js.map