"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getLocalCryptographicMaterialsCache = void 0;
const lru_cache_1 = __importDefault(require("lru-cache"));
const material_management_1 = require("@aws-crypto/material-management");
function getLocalCryptographicMaterialsCache(capacity, proactiveFrequency = 1000 * 60) {
    const cache = new lru_cache_1.default({
        max: capacity,
        dispose(_key, value) {
            /* Zero out the unencrypted dataKey, when the material is removed from the cache. */
            value.response.zeroUnencryptedDataKey();
        },
    });
    (function proactivelyTryAndEvictTail() {
        const timeout = setTimeout(() => {
            mayEvictTail();
            proactivelyTryAndEvictTail();
        }, proactiveFrequency);
        /* In Node.js the event loop will _only_ exit if there are no outstanding events.
         * This means that if I did nothing the event loop would *always* be blocked.
         * This is unfortunate and very bad for things like Lambda.
         * So, I tell Node.js to not wait for this timer.
         * See: https://nodejs.org/api/timers.html#timers_timeout_unref
         */
        // @ts-ignore
        timeout.unref && timeout.unref();
    })();
    return {
        putEncryptionMaterial(key, material, plaintextLength, maxAge) {
            /* Precondition: putEncryptionMaterial plaintextLength can not be negative. */
            (0, material_management_1.needs)(plaintextLength >= 0, 'Malformed plaintextLength');
            /* Precondition: Only cache EncryptionMaterial. */
            (0, material_management_1.needs)((0, material_management_1.isEncryptionMaterial)(material), 'Malformed response.');
            /* Precondition: Only cache EncryptionMaterial that is cacheSafe. */
            (0, material_management_1.needs)(material.suite.cacheSafe, 'Can not cache non-cache safe material');
            const entry = Object.seal({
                response: material,
                bytesEncrypted: plaintextLength,
                messagesEncrypted: 1,
                now: Date.now(),
            });
            cache.set(key, entry, maxAge);
        },
        putDecryptionMaterial(key, material, maxAge) {
            /* Precondition: Only cache DecryptionMaterial. */
            (0, material_management_1.needs)((0, material_management_1.isDecryptionMaterial)(material), 'Malformed response.');
            /* Precondition: Only cache DecryptionMaterial that is cacheSafe. */
            (0, material_management_1.needs)(material.suite.cacheSafe, 'Can not cache non-cache safe material');
            const entry = Object.seal({
                response: material,
                bytesEncrypted: 0,
                messagesEncrypted: 0,
                now: Date.now(),
            });
            cache.set(key, entry, maxAge);
        },
        putBranchKeyMaterial(key, material, maxAge) {
            /* Precondition: Only cache BranchKeyMaterial */
            (0, material_management_1.needs)((0, material_management_1.isBranchKeyMaterial)(material), 'Malformed response.');
            const entry = Object.seal({
                response: material,
                now: Date.now(),
            });
            cache.set(key, entry, maxAge);
        },
        getEncryptionMaterial(key, plaintextLength) {
            /* Precondition: plaintextLength can not be negative. */
            (0, material_management_1.needs)(plaintextLength >= 0, 'Malformed plaintextLength');
            const entry = cache.get(key);
            /* Check for early return (Postcondition): If this key does not have an EncryptionMaterial, return false. */
            if (!entry)
                return false;
            /* Postcondition: Only return EncryptionMaterial. */
            (0, material_management_1.needs)((0, material_management_1.isEncryptionMaterial)(entry.response), 'Malformed response.');
            const encryptionMaterialEntry = entry;
            encryptionMaterialEntry.bytesEncrypted += plaintextLength;
            encryptionMaterialEntry.messagesEncrypted += 1;
            return entry;
        },
        getDecryptionMaterial(key) {
            const entry = cache.get(key);
            /* Check for early return (Postcondition): If this key does not have a DecryptionMaterial, return false. */
            if (!entry)
                return false;
            /* Postcondition: Only return DecryptionMaterial. */
            (0, material_management_1.needs)((0, material_management_1.isDecryptionMaterial)(entry.response), 'Malformed response.');
            return entry;
        },
        getBranchKeyMaterial(key) {
            const entry = cache.get(key);
            /* Postcondition: If this key does not have a BranchKeyMaterial, return false */
            if (!entry)
                return false;
            /* Postcondition: Only return BranchKeyMaterial */
            (0, material_management_1.needs)((0, material_management_1.isBranchKeyMaterial)(entry.response), 'Malformed response.');
            return entry;
        },
        del(key) {
            cache.del(key);
        },
    };
    function mayEvictTail() {
        // @ts-ignore
        const { tail } = cache.dumpLru();
        /* Check for early return (Postcondition) UNTESTED: If there is no tail, then the cache is empty. */
        if (!tail)
            return;
        /* The underlying Yallist tail Node has a `value`.
         * This value is a lru-cache Entry and has a `key`.
         */
        const { key } = tail.value;
        // Peek will evict, but not update the "recently used"-ness of the key.
        cache.peek(key);
    }
}
exports.getLocalCryptographicMaterialsCache = getLocalCryptographicMaterialsCache;
//# sourceMappingURL=data:application/json;base64,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